/*!
 * @file        app.c
 *
 * @brief       Application processing program body
 *
 * @version     V1.0.0
 *
 * @date        2022-02-25
 *
 * @attention
 *
 *  Copyright (C) 2021-2022 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be usefull and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

#include "app.h"
#include "bsp_touch.h"
#include "bsp_i2c.h"
#include "bsp_delay.h"
#include "picture.h"
#include "bsp_lcd.h"
#include "bsp_lcdFont.h"
#include <stdio.h>

TOUCH_EVENT_T touchLED;
TOUCH_EVENT_T touchRefresh;
TOUCH_EVENT_T touchHelp;

static uint8_t ledStatus = 0;

/*!
 * @brief       Touch board display initial UI
 *
 * @param       None
 *
 * @retval      None
 *
 */
void Touch_DisplayInitUI(void)
{
    LCD_DisplayPic(260,100,240,108,imageGeehyLogo);
    
    LCD_DisplayPic(250,220,50,50,imageLedOFF);
    
    LCD_DisplayPic(350,220,50,50,imageRefresh);
    
    LCD_DisplayPic(450,220,50,50,imageHelp);
    
    LCD_PrintString(30, 10, "APM32F407 EVAL Touch Example!", LCD_COLOR_USER);
}

/*!
 * @brief       Touch board help button event handle
 *
 * @param       None
 *
 * @retval      None
 *
 */
void Touch_HelpHandle(void)
{
    LCD_PrintString(270, 300, "www.geehy.com", LCD_COLOR_BLUE);
}

/*!
 * @brief       Touch board refresh button event handle
 *
 * @param       None
 *
 * @retval      None
 *
 */
void Touch_RefreshHandle(void)
{
    LCD_ClearScreen();
    
    Touch_DisplayInitUI();
    
    APM_EVAL_LEDOff(LED2);
    APM_EVAL_LEDOff(LED3);
    
    ledStatus = 0;
}

/*!
 * @brief       Touch board led button event handle
 *
 * @param       status: led button touch status
 *
 * @retval      None
 *
 */
void Touch_LedHandle(uint8_t status)
{
    if(status)
    {
        LCD_DisplayPic(250,220,50,50,imageLedON);
        APM_EVAL_LEDOn(LED2);
        APM_EVAL_LEDOn(LED3);
    }
    else
    {
        LCD_DisplayPic(250,220,50,50,imageLedOFF);
        APM_EVAL_LEDOff(LED2);
        APM_EVAL_LEDOff(LED3);
    }
}

/*!
 * @brief       Touch board event handle
 *
 * @param       None
 *
 * @retval      None
 *
 */
void Touch_EventHandle(void)
{
    /** touch help button down event*/
    if(touchHelp.touchDownStatus)
    {
        LCD_DisplayPic(450,220,50,50,imageHelp2);
    }
    
    /** touch help button up event*/
    if(touchHelp.touchUpStatus)
    {
        LCD_DisplayPic(450,220,50,50,imageHelp);
    }
    
    /** touch refresh button down event*/
    if(touchRefresh.touchDownStatus)
    {
        LCD_DisplayPic(350,220,50,50,imageRefresh2);
    }
    
    /** touch refresh button up event*/
    if(touchRefresh.touchUpStatus)
    {
        LCD_DisplayPic(350,220,50,50,imageRefresh);
    }
    
    if(ledStatus)
    {
        /** touch led button down event*/
        if(touchLED.touchDownStatus)
        {
            LCD_DisplayPic(250,220,50,50,imageLedON2);
        }
        
        /** touch led button up event*/
        if(touchLED.touchUpStatus)
        {
            LCD_DisplayPic(250,220,50,50,imageLedON);
        }
    }
    else
    {
        /** touch led button down event*/
        if(touchLED.touchDownStatus)
        {
            LCD_DisplayPic(250,220,50,50,imageLedOFF2);
        }
        
        /** touch led button up event*/
        if(touchLED.touchUpStatus)
        {
            LCD_DisplayPic(250,220,50,50,imageLedOFF);
        }
    }
    
    /** touch led button click event*/
    if(touchLED.touchDownStatus && touchLED.touchUpStatus)
    {
        touchLED.touchDownStatus = 0;
        touchLED.touchUpStatus = 0;
        
        ledStatus ^= 1;
        
        Touch_LedHandle(ledStatus);
    }
    
    /** touch help button click event*/
    if(touchHelp.touchDownStatus && touchHelp.touchUpStatus)
    {
        touchHelp.touchDownStatus = 0;
        touchHelp.touchUpStatus = 0;

        Touch_HelpHandle();
    }
    
    /** touch refresh button click event*/
    if(touchRefresh.touchDownStatus && touchRefresh.touchUpStatus)
    {
        touchRefresh.touchDownStatus = 0;
        touchRefresh.touchUpStatus = 0;
        
        Touch_RefreshHandle();
    }
}
